import stringSimilarity from "string-similarity";

interface Api {
  sendMessage(message: string | object, threadID: string, callback?: (err: any, res: any) => void, messageID?: string): Promise<any>;
  setMessageReactionMqtt(emoji: string, messageID: string, threadID: string): Promise<any>;
  addUserToGroup(uid: string, threadID: string): Promise<any>;
  removeUserFromGroup(uid: string, threadID: string): Promise<any>;
}

interface Models {
  // Define model types if known
}

interface UsersController {
  // Define Users controller methods if known
}

interface ThreadsController {
  getInfo(threadID: string): Promise<{ adminIDs: { id: string }[] }>;
}

interface CurrenciesController {
  // Define Currencies controller methods if known
}

interface Event {
  body?: string;
  senderID: string;
  threadID: string;
  messageID: string;
}

interface ZuckbotConfig {
  name: string;
  aliases?: string[];
  version: string;
  author: string;
  role: number;
  info: string;
  Category: string;
  image: string[];
  shadowPrefix: boolean;
  cd: number;
}

interface CommandModule {
  zuckbotconfig: ZuckbotConfig;
  onRun: (options: {
    api: Api;
    event: Event;
    args: string[];
    models: Models;
    Users: UsersController;
    Threads: ThreadsController;
    Currencies: CurrenciesController;
    role: number;
    send: (msg: string | object, callback?: (err: any, res: any) => void) => Promise<any>;
    sendVD: (msg: string | object, callback?: (err: any, res: any) => void) => Promise<any>;
    box: Box;
    output: (a: any) => Promise<any>;
  }) => void;
}

interface Box {
  react(emoji: string): Promise<any>;
  reply(msg: string | object): Promise<any>;
  add(uid: string): Promise<any>;
  kick(uid: string): Promise<any>;
  send(msg: string | object, id?: string): Promise<any>;
}

const escapeRegex = (str: string): string => str.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");

const findCommand = (commandName: string, commands: Map<string, CommandModule>): CommandModule | null => {
  for (let [name, command] of commands) {
    if (name === commandName || (command.zuckbotconfig.aliases && command.zuckbotconfig.aliases.includes(commandName))) {
      return command;
    }
  }
  return null;
};

export default function ({ api, models, Users, Threads, Currencies }: { api: Api; models: Models; Users: UsersController; Threads: ThreadsController; Currencies: CurrenciesController }) {
  return async function ({ event }: { event: Event }) {
    const dateNow = Date.now();
    const { botPrefix, botAdmins } = global.settings;
    const { threadData } = global.db;
    const { commands, cd, queues } = global.zuckbot;
    const { body = "x", senderID: rawSenderID, threadID: rawThreadID, messageID } = event;
    const senderID = String(rawSenderID);
    const threadID = String(rawThreadID);
    const threadSetting = threadData.get(threadID) || {};
    const prefix = threadSetting.hasOwnProperty("boxPrefix") ? threadSetting.boxPrefix : botPrefix;
    const prefixRegex = new RegExp(`^(<@!?${senderID}>|${escapeRegex(prefix)})\\s*`);

    const send = (msg: string | object, callback?: (err: any, res: any) => void) => new Promise<any>(r =>
      api.sendMessage(msg, threadID, (err: any, res: any) => {
        if (callback) callback(err, res);
        r(res || err);
      },
        messageID
      )
    );

    const uptime = process.uptime();
    const hours = Math.floor(uptime / 3600);
    const minutes = Math.floor((uptime % 3600) / 60);
    const seconds = Math.floor(uptime % 60);

    const [matchedPrefix] = body.match(prefixRegex) || [""];
    let args = body.slice(matchedPrefix.length).trim().split(/ +/);
    let commandName = args.shift()?.toLowerCase();
    let command = commandName ? findCommand(commandName, commands) : null;

    if (!prefixRegex.test(body)) {
      args = (body || "").trim().split(/ +/);
      commandName = args.shift()?.toLowerCase();
      command = commandName ? findCommand(commandName, commands) : null;
      if (command && command.zuckbotconfig) {
        if (command.zuckbotconfig.shadowPrefix === false) {
          if (commandName && commandName.toLowerCase() !== command.zuckbotconfig.name.toLowerCase() && !(command.zuckbotconfig.aliases || []).includes(commandName)) {
            api.sendMessage(command.zuckbotconfig.name, threadID, messageID);
            return;
          }
        } else if (command.zuckbotconfig.shadowPrefix === true && !body.startsWith(botPrefix)) {
          return;
        }
      }
    }

    if (!command) {
      if (!body.startsWith(prefix)) return;
      const allCommandNames = Array.from(commands.keys()).concat(
        ...Array.from(commands.values()).flatMap(cmd => cmd.zuckbotconfig.aliases || [])
      );
      const bestMatch = stringSimilarity.findBestMatch(commandName || "", allCommandNames).bestMatch;
      if (bestMatch.rating >= 0.5) {
        command = findCommand(bestMatch.target, commands);
      }
      if (queues.length === 0) {
        return send(`Lệnh gợi ý "${bestMatch.target}"\nupt: ${hours}:${minutes}:${seconds}`);
      } else {
        return send({ body: `Lệnh gợi ý "${bestMatch.target}"\nupt: ${hours}:${minutes}:${seconds}`, attachment: queues.splice(0, 1) });
      }
    }

    if (command) {
      const threadInfo = await Threads.getInfo(threadID);
      const isAdmin = threadInfo.adminIDs.some(admin => admin.id === senderID);
      let role = 0;
      if (botAdmins.includes(senderID)) role = 2;
      else if (isAdmin) role = 1;

      if (command.zuckbotconfig.role > 2) command.zuckbotconfig.role = 2;
      if (command.zuckbotconfig.role > role) {
        return api.sendMessage(`Bạn không đủ quyền hạn để sử dụng lệnh ${command.zuckbotconfig.name}`, threadID, messageID);
      }

      if (!cd.has(command.zuckbotconfig.name)) {
        cd.set(command.zuckbotconfig.name, new Map<string, number>());
      }
      const timestamps = cd.get(command.zuckbotconfig.name);
      const expirationTime = (command.zuckbotconfig.cd || 1) * 1000;

      if (timestamps.has(senderID) && dateNow < (timestamps.get(senderID) || 0) + expirationTime) {
        return api.sendMessage(`🔄 Vui lòng quay lại sau ${(((timestamps.get(senderID) || 0) + expirationTime - dateNow) / 1000).toFixed(2)}s`, threadID, messageID);
      }

      const sendVD = (msg: string | object, callback?: (err: any, res: any) => void) => new Promise<any>(r =>
        api.sendMessage(
          { body: msg, attachment: queues.splice(0, 1) },
          threadID,
          (err: any, res: any) => {
            if (callback) callback(err, res);
            r(res || err);
          },
          messageID
        )
      );

      const box: Box = {
        react: (emoji: string) => api.setMessageReactionMqtt(emoji, messageID, threadID),
        reply: (msg: string | object) => api.sendMessage(msg, threadID, messageID),
        add: (uid: string) => api.addUserToGroup(uid, threadID),
        kick: (uid: string) => api.removeUserFromGroup(uid, threadID),
        send: (msg: string | object, id?: string) => api.sendMessage(msg, id || threadID),
      };

      const output = (a: any) => {
        if (typeof a === "object" || Array.isArray(a)) {
          if (Object.keys(a).length !== 0) a = JSON.stringify(a, null, 4);
          else a = ""; // Handle empty objects/arrays
        }
        if (typeof a === "number") a = a.toString();
        return api.sendMessage(a, threadID, messageID);
      };

      try {
        command.onRun({ api, event, args, models, Users, Threads, Currencies, role, send, sendVD, box, output });
        timestamps.set(senderID, dateNow);
      } catch (e: any) {
        return api.sendMessage(`Đã xảy ra lỗi không mong muốn trong khi thực thi lệnh ${commandName}, lỗi ${e.message || e}`, threadID, messageID);
      }
    }
  };
}


