import * as fs from "fs-extra";
import axios from "axios";
import { join } from "path";

// Define interfaces for better type safety
interface Api {
  getCurrentUserID(): string;
  sendMessage(message: string | object, threadID: string, messageID?: string): Promise<any>;
  unsendMessage(messageID: string): Promise<any>;
  getThreadInfo(threadID: string): Promise<any>;
  getUserInfo(userID: string): Promise<any>;
}

interface Models {
  use(modelName: string): any; // Replace 'any' with specific model types if available
}

interface UserData {
  userID: string;
  name: string;
  data: object;
}

interface ThreadData {
  threadID: string;
  threadInfo: object;
  data: object;
}

interface Event {
  type: string;
  body?: string;
  senderID: string;
  threadID: string;
  messageID: string;
  logMessageData?: any;
  logMessageType?: string;
  logMessageBody?: string;
}

interface Controllers {
  Users: any;
  Threads: any;
  Currencies: any;
}

export default function ({ api, models }: { api: Api; models: Models }) {
  const Users = require(join(global.zuckbot.mainPath, "db/controllers/users")).default({ models, api });
  const Threads = require(join(global.zuckbot.mainPath, "db/controllers/threads")).default({ models, api });
  const Currencies = require(join(global.zuckbot.mainPath, "db/controllers/currencies")).default({ models });

  const logger = global.loading;

  (async function () {
    try {
      const [threads, users] = await Promise.all([
        Threads.getAll() as Promise<ThreadData[]>,
        Users.getAll(["userID", "name", "data"]) as Promise<UserData[]>,
      ]);

      threads.forEach((data: ThreadData) => {
        const idThread = String(data.threadID);
        global.db.allThreadID.push(idThread);
        global.db.threadData.set(idThread, data.data || {});
        global.db.threadInfo.set(idThread, data.threadInfo || {});
      });

      users.forEach((dataU: UserData) => {
        const idUsers = String(dataU.userID);
        global.db.allUserID.push(idUsers);
        if (dataU.name && dataU.name.length !== 0) {
          global.db.userName.set(idUsers, dataU.name);
        }
      });
    } catch (error) {
      logger(`Can't load environment variable, error: ${error}`, "error");
    }
  })();

  const onRun = require("./onRun").default({ api, models, Users, Threads, Currencies });
  const CommandEvent = require("./CommandEvent").default({ api, models, Users, Threads, Currencies });
  const onReply = require("./onReply").default({ api, models, Users, Threads, Currencies });
  const onReaction = require("./onReaction").default({ api, models, Users, Threads, Currencies });
  const onEvent = require("./onEvent").default({ api, models, Users, Threads, Currencies });
  const onRefresh = require("./onRefresh").default({ api, Threads, Users, Currencies, models });
  const CreateDatabase = require("./CreateDatabase").default({ api, Threads, Users, Currencies, models });
  require("./onSchedule.js").default({ api, Threads, Users, models }); // Assuming onSchedule will also be converted to TS

  return async (event: Event) => {
    if (
      (event.body || "") &&
      event.senderID !== api.getCurrentUserID() &&
      !global.settings.botAdmins.includes(event.senderID)
    ) {
      const thuebotPath = join(process.cwd(), "db/databot/thuebot.json");
      const thuebot = JSON.parse(fs.readFileSync(thuebotPath, "utf8"));
      const c = thuebot.find(($: any) => $.t_id === event.threadID);
      if (!c) return;
    }

    if (event.type === "message" || event.type === "message_reply" || event.type === "message_unsend") {
      CreateDatabase({ event });
      onRun({ event });
      onReply({ event });
      CommandEvent({ event });
    } else if (event.type === "change_thread_image") {
      // Handle change_thread_image event if needed
    } else if (event.type === "event") {
      onEvent({ event });
      onRefresh({ event });
    } else if (event.type === "message_reaction") {
      const unsendPath = join(process.cwd(), "db/databot/123.json");
      let unsendData: { [key: string]: { data: boolean } };

      if (!fs.existsSync(unsendPath)) {
        fs.writeFileSync(unsendPath, JSON.stringify({}, null, 4));
        unsendData = {};
      } else {
        unsendData = JSON.parse(fs.readFileSync(unsendPath, "utf8"));
      }

      if (!unsendData[event.threadID]) {
        unsendData[event.threadID] = { data: false };
      }
      fs.writeFileSync(unsendPath, JSON.stringify(unsendData, null, 4));

      if (unsendData[event.threadID].data) {
        api.unsendMessage(event.messageID);
      }
      onReaction({ event });
    } else {
      // Handle other event types if needed
    }
  };
}


