import { Model } from "sequelize";

interface UsersModel extends Model {
  userID: number;
  name: string;
  data: object;
}

export default function ({ models, api }: { models: any; api: any }) {
  const Users: { new (): UsersModel } & typeof Model = models.use("Users");

  async function getInfo(id: number): Promise<any> {
    return (await api.getUserInfo(id))[id];
  }

  async function getNameUser(id: number): Promise<string> {
    try {
      if (global.db.userName.has(id)) return global.db.userName.get(id);
      else if (global.db.allUserID.includes(id)) {
        const nameUser = (await getData(id)).name;
        if (nameUser) return nameUser;
        else return "Người dùng facebook";
      } else return "Người dùng facebook";
    } catch {
      return "Người dùng facebook";
    }
  }

  async function getAll(...data: (object | string[])[]): Promise<any[]> {
    let where: object | undefined;
    let attributes: string[] | undefined;

    for (const i of data) {
      if (typeof i !== "object") throw new Error("Phải là một Array hoặc Object hoặc có thể là cả hai");
      if (Array.isArray(i)) attributes = i;
      else where = i;
    }
    try {
      return (await Users.findAll({ where, attributes })).map((e) =>
        e.get({ plain: true }),
      );
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function getData(userID: number): Promise<any | false> {
    try {
      const data = await Users.findOne({ where: { userID } });
      if (data) return data.get({ plain: true });
      else return false;
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function setData(userID: number, options: object = {}): Promise<boolean> {
    if (typeof options !== "object" && !Array.isArray(options)) throw new Error("Phải là một Object");
    try {
      const user = await Users.findOne({ where: { userID } });
      if (user) {
        await user.update(options);
        return true;
      } else {
        await createData(userID, options);
        return true;
      }
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function delData(userID: number): Promise<boolean> {
    try {
      const user = await Users.findOne({ where: { userID } });
      if (user) {
        await user.destroy();
        return true;
      }
      return false; // User not found
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function createData(userID: number, defaults: object = {}): Promise<boolean> {
    if (typeof defaults !== "object" && !Array.isArray(defaults)) throw new Error("needObject");
    try {
      await Users.findOrCreate({ where: { userID }, defaults });
      return true;
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  return {
    getInfo,
    getNameUser,
    getAll,
    getData,
    setData,
    delData,
    createData,
  };
}


