import { Model } from "sequelize";

interface ThreadsModel extends Model {
  threadID: number;
  threadInfo: object;
  data: object;
}

export default function ({ models, api }: { models: any; api: any }) {
  const Threads: { new (): ThreadsModel } & typeof Model = models.use("Threads");

  async function getInfo(threadID: number): Promise<any> {
    try {
      const result = await api.getThreadInfo(threadID);
      return result;
    } catch (error) {
      console.log(error);
      throw new Error(error as string);
    }
  }

  async function getAll(...data: (object | string[])[]): Promise<any[]> {
    let where: object | undefined;
    let attributes: string[] | undefined;

    for (const i of data) {
      if (typeof i !== "object") throw new Error("Phải là một Array hoặc Object hoặc có thể là cả hai");
      if (Array.isArray(i)) attributes = i;
      else where = i;
    }
    try {
      return (await Threads.findAll({ where, attributes })).map((e) =>
        e.get({ plain: true }),
      );
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function getData(threadID: number): Promise<any | false> {
    try {
      const data = await Threads.findOne({ where: { threadID } });
      if (data) return data.get({ plain: true });
      else return false;
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function setData(threadID: number, options: object = {}): Promise<boolean> {
    if (typeof options !== "object" && !Array.isArray(options)) throw new Error("Phải là một Object");
    try {
      const thread = await Threads.findOne({ where: { threadID } });
      if (thread) {
        await thread.update(options);
        return true;
      } else {
        await createData(threadID, options);
        return true;
      }
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function delData(threadID: number): Promise<boolean> {
    try {
      const thread = await Threads.findOne({ where: { threadID } });
      if (thread) {
        await thread.destroy();
        return true;
      }
      return false; // Thread not found
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function createData(threadID: number, defaults: object = {}): Promise<boolean> {
    if (typeof defaults !== "object" && !Array.isArray(defaults)) throw new Error("Phải là một Object");
    try {
      await Threads.findOrCreate({ where: { threadID }, defaults });
      return true;
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  return {
    getInfo,
    getAll,
    getData,
    setData,
    delData,
    createData,
  };
}


