import { Model } from "sequelize";

interface CurrenciesModel extends Model {
  userID: number;
  coins: number;
  exp: number;
  data: object;
}

export default function ({ models }: { models: any }) {
  const Currencies: { new (): CurrenciesModel } & typeof Model = models.use("Currencies");

  async function getAll(...data: (object | string[])[]): Promise<any[]> {
    let where: object | undefined;
    let attributes: string[] | undefined;

    for (const i of data) {
      if (typeof i !== "object") throw new Error("Phải là một Array hoặc Object hoặc có thể là cả hai");
      if (Array.isArray(i)) attributes = i;
      else where = i;
    }
    try {
      return (await Currencies.findAll({ where, attributes })).map((e) =>
        e.get({ plain: true }),
      );
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function getData(userID: number): Promise<any | false> {
    try {
      const data = await Currencies.findOne({ where: { userID } });
      if (data) return data.get({ plain: true });
      else return false;
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function setData(userID: number, options: object = {}): Promise<boolean> {
    if (typeof options !== "object" && !Array.isArray(options)) throw new Error("Phải là một Object");
    try {
      const currency = await Currencies.findOne({ where: { userID } });
      if (currency) {
        await currency.update(options);
        return true;
      }
      return false; // User not found
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function delData(userID: number): Promise<boolean> {
    try {
      const currency = await Currencies.findOne({ where: { userID } });
      if (currency) {
        await currency.destroy();
        return true;
      }
      return false; // User not found
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function createData(userID: number, defaults: object = {}): Promise<boolean> {
    if (typeof defaults !== "object" && !Array.isArray(defaults)) throw new Error("Phải là một Array hoặc Object hoặc có thể là cả hai");
    try {
      await Currencies.findOrCreate({ where: { userID }, defaults });
      return true;
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function increaseCoins(userID: number, coins: number): Promise<boolean> {
    if (typeof coins !== "number") throw new Error("Phải là một con số");
    try {
      const data = await getData(userID);
      if (data) {
        await setData(userID, { coins: data.coins + coins });
        return true;
      }
      return false; // User not found
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  async function decreaseCoins(userID: number, coins: number): Promise<boolean> {
    if (typeof coins !== "number") throw new Error("Phải là một con số");
    try {
      const data = await getData(userID);
      if (data) {
        if (data.coins < coins) return false;
        await setData(userID, { coins: data.coins - coins });
        return true;
      }
      return false; // User not found
    } catch (error) {
      console.error(error);
      throw new Error(error as string);
    }
  }

  return {
    getAll,
    getData,
    setData,
    delData,
    createData,
    increaseCoins,
    decreaseCoins,
  };
}


